/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * license agreements; and to You under the Apache License, version 2.0:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * This file is part of the Apache Pekko project, which was derived from Akka.
 */

/*
 * Copyright (C) 2009-2022 Lightbend Inc. <https://www.lightbend.com>
 */

package docs.http.javadsl;

import org.junit.Test;
import static org.junit.Assert.assertEquals;

import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import com.fasterxml.jackson.annotation.JsonRootName;

import org.apache.pekko.http.javadsl.testkit.JUnitRouteTest;
import org.apache.pekko.http.javadsl.model.*;
import org.apache.pekko.http.javadsl.server.Route;
import org.apache.pekko.http.javadsl.unmarshalling.Unmarshaller;

import static org.apache.pekko.http.javadsl.server.Directives.*;

public class JacksonXmlExampleTest extends JUnitRouteTest {

  final String xml = "<point><x>3</x><y>4</y></point>";
  final Point point =
      new Point() {
        {
          setX(3);
          setY(4);
        }
      };

  @Test
  public void marshalXml() throws Exception {
    final Route route = concat(completeOK(point, JacksonXmlSupport.<Point>marshaller()));

    testRoute(route).run(HttpRequest.GET("/")).assertStatusCode(StatusCodes.OK).assertEntity(xml);
  }

  @Test
  public void unmarshalXml() throws Exception {
    final Unmarshaller<HttpEntity, Point> unmarshaller =
        JacksonXmlSupport.unmarshaller(Point.class);

    final Route route =
        concat(
            entity(
                unmarshaller,
                p -> {
                  assertEquals(p, point);
                  return complete(p.toString());
                }));

    final RequestEntity entity = HttpEntities.create(ContentTypes.TEXT_XML_UTF8, xml);

    testRoute(route)
        .run(HttpRequest.POST("/").withEntity(entity))
        .assertStatusCode(StatusCodes.OK)
        .assertEntity("Point(x=3, y=4)");
  }

  @Test
  public void unmarshalXmlDirect() throws Exception {
    {
      CompletionStage<Point> resultStage =
          JacksonXmlSupport.unmarshaller(Point.class)
              .unmarshal(HttpEntities.create(ContentTypes.TEXT_XML_UTF8, xml), system());

      assertEquals(point, resultStage.toCompletableFuture().get(3, TimeUnit.SECONDS));
    }

    {
      CompletionStage<Point> resultStage =
          JacksonXmlSupport.unmarshaller(Point.class)
              .unmarshal(
                  HttpEntities.create(
                      ContentTypes.create(MediaTypes.APPLICATION_XML, HttpCharsets.UTF_8), xml),
                  system());

      assertEquals(point, resultStage.toCompletableFuture().get(3, TimeUnit.SECONDS));
    }
  }

  @JsonRootName("point")
  public static class Point {
    private int x, y;

    public void setX(int x) {
      this.x = x;
    }

    public int getX() {
      return this.x;
    }

    public void setY(int y) {
      this.y = y;
    }

    public int getY() {
      return this.y;
    }

    public String toString() {
      return "Point(x=" + x + ", y=" + y + ")";
    }

    public boolean equals(Object other) {
      Point that = other instanceof Point ? (Point) other : null;
      return that != null && that.x == this.x && that.y == this.y;
    }
  }
}
